ShapeMaker = class()

function ShapeMaker:init()
    self.restitution = 0.25
    self.sleepingAllowed = false
    self.interpolate = true -- enables smooth motion
end

function ShapeMaker:circleAt(x, y, radius)
    local circle = physics.body(CIRCLE, radius)
    circle.x = x
    circle.y = y
    circle.restitution = self.restitution
    circle.sleepingAllowed = self.sleepingAllowed
    circle.interpolate = self.interpolate
    return circle
end

function ShapeMaker:polygonAt(x, y, pointsTable)
    -- polygons are defined by a series of points in counter-clockwise order
    local poly = physics.body(POLYGON, table.unpack(pointsTable))
    poly.x = x
    poly.y = y
    poly.restitution = self.restitution
    poly.sleepingAllowed = self.sleepingAllowed
    return poly
end

function ShapeMaker:boxAt(x, y, w, h)
    local pointsTable = {vec2(-w/2,h/2), vec2(-w/2,-h/2), vec2(w/2,-h/2), vec2(w/2,h/2)}
    local box = self:polygonAt(x, y, pointsTable)
    box.interpolate = true
    return box
end

function ShapeMaker:randomPolygonAt(x, y, minSize, maxSize)
    minSize = minSize or 25
    maxSize = maxSize or 75
    local points = {}
    local numSides = math.random(3,10)
    local radius = math.random(math.floor(minSize), math.floor(maxSize))
    local offsetRange = math.floor(radius * 0.3)
    local direction = 0
    local directionIncrement = 2 * math.pi / numSides
    
    for i = 1, numSides do
        local distanceFromCenter = vec2(radius,0)
        local pointOnCircumference = distanceFromCenter:rotate(direction)
        local randomOffset = vec2(math.random(offsetRange * -1, offsetRange), math.random(offsetRange * -1, offsetRange))
        local newPoint = pointOnCircumference + randomOffset
        direction = direction + directionIncrement
        table.insert(points, newPoint)
    end
    
    local poly = self:polygonAt(x, y, points)
    return poly
end

function ShapeMaker:pointsTableForSineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    local ang = math.atan(point2.y - point1.y, point2.x - point1.x) * 180 / math.pi
    local dist = vec2(point1.x, point1.y):dist(vec2(point2.x, point2.y))
    numCycles = numCycles or 5
    waveHeight = waveHeight or dist / 5
    segmentLength = segmentLength or 1
    local spaceBetweenPeaks = dist / (numCycles * 360)
    local points = {}
    for z=0, 360*numCycles, segmentLength do
        x2=z
        y2=math.sin(math.rad(z))*waveHeight
        x1=x2*spaceBetweenPeaks*math.cos(math.rad(ang))-y2*math.sin(math.rad(ang))
        y1=y2*math.cos(math.rad(ang))+x2*spaceBetweenPeaks*math.sin(math.rad(ang))
        x1= x1 + point1.x
        y1 = y1 + point1.y
        table.insert(points, vec2(x1,y1))
    end
    return points
end

function ShapeMaker:collidableWall(x1, y1, x2, y2)
    return physics.body(EDGE, vec2(x1, y1), vec2(x2, y2))
end

function ShapeMaker:CHAIN_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    local points = self:pointsTableForSineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    return physics.body(CHAIN, false, table.unpack(points))
end

function ShapeMaker:POLYGON_SineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    local points = self:pointsTableForSineWaveBetweenPoints(point1, point2, numCycles, waveHeight, segmentLength)
    return physics.body(POLYGON, table.unpack(points))
end


